#----------------------------------------------------------------------
#  GFDM method test - 2d heat equation, Dirichlet BC, Implicit Euler
#  Author: Andrea Pavan
#  Date: 15/12/2022
#  License: GPLv3-or-later
#----------------------------------------------------------------------
using ElasticArrays;
using LinearAlgebra;
using SparseArrays;
using Printf;
using PyPlot;
include("utils.jl");


#problem definition
l1 = 1.0;       #domain x size
l2 = 0.1;       #domain y size
uL = 400;       #left border temperature
uR = 300;       #right border temperature
kcost = 5.0;        #thermal conductivity
rho = 8.0;      #density
ccost = 3.8;        #specific heat capacity
t0 = 0.0;       #starting time
tend = 120.0;        #ending time
dt = 5.0;       #time step
u0(x,y) = 0.0;      #initial conditions
uD(x,y) = uL+x*(uR-uL)/l1;      #Dirichlet boundary conditions

meshSize = 0.02;        #distance target between internal nodes
surfaceMeshSize = 0.02;        #distance target between boundary nodes
minNeighbors = 8;       #minimum number of neighbors allowed
minSearchRadius = meshSize/2;       #starting search radius


#pointcloud generation
time1 = time();
pointcloud = ElasticArray{Float64}(undef,2,0);      #2xN matrix containing the coordinates [X;Y] of each node
boundaryNodes = Vector{Int};        #indices of the boundary nodes
dirichletNodes = Vector{Int};       #indices of the Dirichlet nodes
neumannNodes = Vector{Int};     #indices of the Neumann nodes
normals = ElasticArray{Float64}(undef,2,0);     #2xN matrix containing the components [nx;ny] of the normal of each boundary node
for i=0:surfaceMeshSize:l1
    append!(pointcloud, [i,0.0]);
    append!(normals, [0,-1]);
end
for i=surfaceMeshSize:surfaceMeshSize:l2
    append!(pointcloud, [l1,i]);
    append!(normals, [1,0]);
end
for i=l1-surfaceMeshSize:-surfaceMeshSize:0
    append!(pointcloud, [i,l2]);
    append!(normals, [0,1]);
end
for i=l2-surfaceMeshSize:-surfaceMeshSize:surfaceMeshSize
    append!(pointcloud, [0,i]);
    append!(normals, [-1,0]);
end
boundaryNodes = collect(range(1,size(pointcloud,2)));
for i=0:meshSize:l1
    for j=0:meshSize:l2
        if i>0 && i<l1 && j>0 && j<l2
            #append!(pointcloud, [i,j]);
            append!(pointcloud, [i,j]+(rand(Float64,2).-0.5).*meshSize/5);
        end
    end
end
internalNodes = collect(range(1+length(boundaryNodes),size(pointcloud,2)));
dirichletNodes = vcat(findall(pointcloud[1,boundaryNodes].==0), findall(pointcloud[1,boundaryNodes].==l1));
neumannNodes = findall((pointcloud[1,boundaryNodes].>0).*(pointcloud[1,boundaryNodes].<l1));
println("Generated pointcloud in ", round(time()-time1,digits=2), " s");
println("Pointcloud properties:");
println("  Boundary nodes: ",length(boundaryNodes));
println("  Internal nodes: ",length(internalNodes));
println("  Memory: ",memoryUsage(pointcloud,boundaryNodes));

#pointcloud plot
figure(1);
plot(pointcloud[1,dirichletNodes],pointcloud[2,dirichletNodes],"r.");
plot(pointcloud[1,neumannNodes],pointcloud[2,neumannNodes],"b.");
plot(pointcloud[1,internalNodes],pointcloud[2,internalNodes],"k.");
title("Pointcloud plot");
axis("equal");
display(gcf());


#neighbor search
time2 = time();
N = size(pointcloud,2);     #number of nodes
neighbors = Vector{Vector{Int}}(undef,N);       #vector containing N vectors of the indices of each node neighbors
Nneighbors = zeros(Int,N);      #number of neighbors of each node
for i=1:N
    searchradius = minSearchRadius;
    while Nneighbors[i]<minNeighbors
        neighbors[i] = Int[];
        #check every other node
        for j=1:N
            if j!=i && all(abs.(pointcloud[:,j]-pointcloud[:,i]).<searchradius)
                push!(neighbors[i],j);
            end
        end
        unique!(neighbors[i]);
        Nneighbors[i] = length(neighbors[i]);
        searchradius += minSearchRadius/2;
    end
end
println("Found neighbors in ", round(time()-time2,digits=2), " s");
println("Connectivity properties:");
println("  Max neighbors: ",maximum(Nneighbors)," (at index ",findfirst(isequal(maximum(Nneighbors)),Nneighbors),")");
println("  Avg neighbors: ",round(sum(Nneighbors)/length(Nneighbors),digits=2));
println("  Min neighbors: ",minimum(Nneighbors)," (at index ",findfirst(isequal(minimum(Nneighbors)),Nneighbors),")");


#neighbors distances and weights
time3 = time();
P = Vector{Array{Float64}}(undef,N);        #relative positions of the neighbors
r2 = Vector{Vector{Float64}}(undef,N);      #relative distances of the neighbors
w2 = Vector{Vector{Float64}}(undef,N);      #neighbors weights
for i=1:N
    P[i] = Array{Float64}(undef,2,Nneighbors[i]);
    r2[i] = Vector{Float64}(undef,Nneighbors[i]);
    w2[i] = Vector{Float64}(undef,Nneighbors[i]);
    for j=1:Nneighbors[i]
        P[i][:,j] = pointcloud[:,neighbors[i][j]]-pointcloud[:,i];
        r2[i][j] = P[i][:,j]'P[i][:,j];
    end
    r2max = maximum(r2[i]);
    for j=1:Nneighbors[i]
        w2[i][j] = exp(-1*r2[i][j]/r2max)^2;
    end
end


#least square matrix inversion
A = Vector{Matrix}(undef,N);        #least-squares matrices
condA = Vector{Float64}(undef,N);       #condition number
B = Vector{Matrix}(undef,N);        #least-squares decomposition matrices
C = Vector{Matrix}(undef,N);        #derivatives coefficients matrices
for i in internalNodes
    xj = P[i][1,:];
    yj = P[i][2,:];
    #=A[i] = [sum(xj.^2 .*w2[i])      sum(xj.*yj.*w2[i])          sum(xj.^3 .*w2[i])          sum(xj.*yj.^2 .*w2[i])      sum(xj.^2 .*yj.*w2[i]);
            sum(xj.*yj.*w2[i])      sum(yj.^2 .*w2[i])          sum(xj.^2 .*yj.*w2[i])      sum(yj.^3 .*w2[i])          sum(xj.*yj.^2 .*w2[i]);
            sum(xj.^3 .*w2[i])      sum(xj.^2 .*yj.*w2[i])      sum(xj.^4 .*w2[i])          sum(xj.^2 .*yj.^2 .*w2[i])  sum(xj.^3 .*yj.*w2[i]);
            sum(xj.*yj.^2 .*w2[i])  sum(yj.^3 .*w2[i])          sum(xj.^2 .*yj.^2 .*w2[i])  sum(yj.^4 .*w2[i])          sum(xj.*yj.^3 .*w2[i]);
            sum(xj.^2 .*yj.*w2[i])  sum(xj.*yj.^2 .*w2[i])      sum(xj.^3 .*yj.*w2[i])      sum(xj.*yj.^3 .*w2[i])      sum(xj.^2 .*yj.^2 .*w2[i])];
    condA[i] = cond(A[i]);
    B[i] = zeros(Float64,5,1+Nneighbors[i]);
    B[i][:,1] = [-sum(xj.*w2[i]); -sum(yj.*w2[i]); -sum(xj.^2 .*w2[i]); -sum(yj.^2 .*w2[i]); -sum(xj.*yj.*w2[i])];
    B[i][1,2:end] = xj.*w2[i];
    B[i][2,2:end] = yj.*w2[i];
    B[i][3,2:end] = xj.^2 .*w2[i];
    B[i][4,2:end] = yj.^2 .*w2[i];
    B[i][5,2:end] = xj.*yj.*w2[i];
    (L,U) = cholesky(A[i]);
    C[i] = inv(U)*inv(L)*B[i];=#

    #error matrix notation
    V = zeros(Float64,Nneighbors[i],5);
    for j=1:Nneighbors[i]
        V[j,:] = [xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j]];
    end
    W = Diagonal(w2[i]);
    A[i] = transpose(V)*W*V;
    condA[i] = cond(A[i]);
    (Q,R) = qr(A[i]);
    C[i] = inv(R)*transpose(Q)*transpose(V)*W;
    #C[i] = inv(transpose(V)*W*V)*transpose(V)*W;
    #C[i] = hcat(-sum(C[i],dims=2), C[i]);
end
for i in neumannNodes
    xj = P[i][1,:];
    yj = P[i][2,:];
    A[i] = [sum(w2[i].*xj.^2)       sum(w2[i].*xj.^3)           sum(w2[i].*xj.*yj.^2)       sum(w2[i].*xj.^2 .*yj);
            sum(w2[i].*xj.^3)       sum(w2[i].*xj.^4)           sum(w2[i].*xj.^2 .*yj.^2)   sum(w2[i].*xj.^3 .*yj);
            sum(w2[i].*xj.*yj.^2)   sum(w2[i].*xj.^2 .*yj.^2)   sum(w2[i].*yj.^4)           sum(w2[i].*xj.*yj.^3);
            sum(w2[i].*xj.^2 .*yj)  sum(w2[i].*xj.^3 .*yj)      sum(w2[i].*xj.*yj.^3)       sum(w2[i].*xj.^2 .*yj.^2)];
    condA[i] = cond(A[i]);
    B[i] = zeros(Float64,4,2+Nneighbors[i]);
    B[i][:,1] = [-sum(xj.*w2[i]); -sum(xj.^2 .*w2[i]); -sum(yj.^2 .*w2[i]); -sum(xj.*yj.*w2[i])];
    B[i][1,2:end-1] = xj.*w2[i];
    B[i][2,2:end-1] = xj.^2 .*w2[i];
    B[i][3,2:end-1] = yj.^2 .*w2[i];
    B[i][4,2:end-1] = xj.*yj.*w2[i];
    B[i][:,end] .= -sum(w2[i].*yj);
    (L,U) = cholesky(A[i]);
    C[i] = inv(U)*inv(L)*B[i];
end
println("Inverted least-squares matrices in ", round(time()-time3,digits=2), " s");
println("Matrices properties:");
println("  Max condition number (internal nodes): ",round(maximum(condA[internalNodes]),digits=2));
println("  Avg condition number (internal nodes): ",round(sum(condA[internalNodes])/length(internalNodes),digits=2));
println("  Min condition number (internal nodes): ",round(minimum(condA[internalNodes]),digits=2));
println("  Max condition number (neumann nodes): ",round(maximum(condA[neumannNodes]),digits=2));
println("  Avg condition number (neumann nodes): ",round(sum(condA[neumannNodes])/length(neumannNodes),digits=2));
println("  Min condition number (neumann nodes): ",round(minimum(condA[neumannNodes]),digits=2));

#condition number plot
figure(2);
scatter(pointcloud[1,internalNodes],pointcloud[2,internalNodes],c=log10.(condA[internalNodes]),cmap="viridis");
scatter(pointcloud[1,neumannNodes],pointcloud[2,neumannNodes],c=log10.(condA[neumannNodes]),cmap="viridis");
colorbar();
title("Condition number (log scale)");
axis("equal");
display(gcf());


#implicit euler matrix assembly
time4 = time();
rows = Int[];
cols = Int[];
vals = Float64[];
for i in dirichletNodes
    push!(rows, i);
    push!(cols, i);
    push!(vals, 1);
end
for i in neumannNodes
    push!(rows, i);
    push!(cols, i);
    push!(vals, 1/dt - (C[i][2,1]+C[i][3,1])*2*kcost/(rho*ccost));
    for j=1:lastindex(neighbors[i])
        push!(rows, i);
        push!(cols, neighbors[i][j]);
        push!(vals, -(C[i][2,1+j]+C[i][3,1+j])*2*kcost/(rho*ccost));
    end
end
for i in internalNodes
    push!(rows, i);
    push!(cols, i);
    #push!(vals, 1/dt - (C[i][3,1]+C[i][4,1])*2*kcost/(rho*ccost));
    push!(vals, 1/dt + (sum(C[i][3,:])+sum(C[i][4,:]))*2*kcost/(rho*ccost));
    for j=1:Nneighbors[i]
        push!(rows, i);
        push!(cols, neighbors[i][j]);
        #push!(vals, -(C[i][3,1+j]+C[i][4,1+j])*2*kcost/(rho*ccost));
        push!(vals, -(C[i][3,j]+C[i][4,j])*2*kcost/(rho*ccost));
    end
end
M = sparse(rows,cols,vals,N,N);
println("Completed matrix assembly in ", round(time()-time4,digits=2), " s");

#matrix plot
figure(3);
spy(M);
title("Crank-Nicolson matrix");
axis("equal");
display(gcf());


#time propagation
time5 = time();
t = collect(t0:dt:tend);        #timesteps
u = u0.(pointcloud[1,:],pointcloud[2,:]);       #numerical solution
uprev = copy(u);        #numerical solution at the previous timestep
ue = uD.(pointcloud[1,:],pointcloud[2,:]);      #exact analytical solution
erru = ue-u;        #numerical solution error
@printf("%6s | %6s | %10s\n","Step","Time","max(err(u))");
@printf("%6i | %6.2f | %10.4e\n",0,t0,maximum(abs.(erru)));
for ti=2:lastindex(t)
    #implicit euler method
    global uprev = copy(u);
    b = zeros(N);       #rhs vector
    for i in dirichletNodes
        b[i] = uD(pointcloud[1,i],pointcloud[2,i]);
    end
    for i in neumannNodes
        b[i] = uprev[i]/dt;
        #=b[i] = uprev[i]*(1/dt + (C[i][2,1]+C[i][3,1])*kcost/(rho*ccost));
        for j=1:lastindex(neighbors[i])
            b[i] += uprev[neighbors[i][j]]*(C[i][2,1+j]+C[i][3,1+j])*kcost/(rho*ccost);
        end=#
    end
    for i in internalNodes
        b[i] = uprev[i]/dt;
        #=b[i] = uprev[i]*(1/dt + (C[i][3,1]+C[i][4,1])*kcost/(rho*ccost));
        for j=1:lastindex(neighbors[i])
            b[i] += uprev[neighbors[i][j]]*(C[i][3,1+j]+C[i][4,1+j])*kcost/(rho*ccost);
        end=#
    end
    global u = M\b;

    #error calculation
    global erru = ue-u;        #numerical solution error
    maxerru = maximum(abs.(erru));
    @printf("%6i | %6.2f | %10.4e\n",ti,t[ti],maxerru);
    if maxerru>1e5
        println("ERROR: the solution diverged");
        break;
    end
end
println("Time integration completed in ", round(time()-time5,digits=2), " s");

#solution plot
figure(4);
scatter(pointcloud[1,:],pointcloud[2,:],c=u,cmap="inferno");
colorbar();
title("Numerical solution");
axis("equal");
display(gcf());

#error plot
figure(5);
scatter(pointcloud[1,:],pointcloud[2,:],c=erru,cmap="viridis");
colorbar();
title("Numerical error distribution");
axis("equal");
display(gcf());
